/* 
   CC0 2011, Martin Haye

   To the extent possible under law, Martin Haye has waived all copyright 
   and related or neighboring rights to p2e: Pseudo-II Emulator. 
   This work is published from: United States.
*/

//
// Support for interpreted mode (non-compiled) 6502 execution.
//

/* ======================================================================= */

// Get value from zero-page 
function v_zp() {
  t += 3;
  return raw_mem[mem_get(pc++)];
}

// Get retrieval pointer to zero-page location 
function p_zp() {
  t += 5;
  return mem_get(pc++);
}

// Get storage pointer to zero-page location 
function s_zp() {
  t += 3;
  return mem_get(pc++);
}

// Get value from zero page addr plus index 
function v_zp_n(add) {
  t += 4;
  return raw_mem[(mem_get(pc++) + add) & 0xFF];
}

// Get retrieval pointer to zero page addr plus index 
function p_zp_n(add) {
  t += 6;
  return (mem_get(pc++) + add) & 0xFF;
}

// Get storage pointer to zero page addr plus index 
function s_zp_n(add) {
  t += 4;
  return (mem_get(pc++) + add) & 0xFF;
}

// Get value at absolute addr 
function v_abs() {
  var addr = mem_get(pc++);
  addr += mem_get(pc++) << 8;
  t += 4;
  if (addr < 0xC000)
    return raw_mem[addr];
  else
    return mem_get(addr);
}

// Get retrieval pointer to absolute addr 
function p_abs() {
  var addr = mem_get(pc++);
  addr += mem_get(pc++) << 8;
  t += 6;
  return addr;
}

// Get storage pointer to absolute addr 
function s_abs() {
  var addr = mem_get(pc++);
  addr += mem_get(pc++) << 8;
  t += 4;
  return addr;
}

// Get value at absolute addr plus index 
function v_abs_n(n) {
  var lo = mem_get(pc++);
  var hi = mem_get(pc++);
  t += 4; // ignore extra cycle for page crossing, to agree with compiled code
  return mem_get(lo + (hi<<8) + n);
}

// Get retreival pointer to absolute addr plus index 
function p_abs_n(n) {
  var addr = mem_get(pc++) + n;
  addr += mem_get(pc++) << 8;
  t += 7;
  return addr & 0xFFFF;
}

// Get storage pointer to absolute addr plus index 
function s_abs_n(n) {
  var addr = mem_get(pc++) + n;
  addr += mem_get(pc++) << 8;
  t += 5;
  return addr & 0xFFFF;
}

// Get value at zero-page (indirect,x) location 
function v_ind_x() {
  t += 6;
  var v_zp = mem_get(pc++) + x;
  return mem_get(mem_get(v_zp) | (mem_get(v_zp+1) << 8));
}

// Get storage pointer to zero-page (indirect,x) location 
function s_ind_x() {
  t += 6;
  var v_zp = mem_get(pc++) + x;
  return mem_get(v_zp) | (mem_get(v_zp+1) << 8);
}

// Get value at zero-page (indirect) location 
function v_ind() {
  t += 5;
  var v_zp = mem_get(pc++);
  return mem_get(mem_get(v_zp) | (mem_get(v_zp+1) << 8));
}

// Get value at memory (indirect),y location 
function v_ind_y() {
  var v_zp = mem_get(pc++);
  var addr = mem_get(v_zp) | (mem_get(v_zp+1) << 8);
  t += 5; // ignore extra cycle for page crossing, to agree with compiled code
  return mem_get(addr + y);
}

// Get storage pointer to memory (indirect),y location 
function s_ind_y() {
  var v_zp = mem_get(pc++);
  var p = (mem_get(v_zp) | (mem_get(v_zp+1) << 8)) + y;
  t += 6;
  return p & 0xFFFF;
}

// Get storage pointer to zero-page (indirect) location 
function s_ind() {
  t += 5;
  var v_zp = mem_get(pc++);
  return mem_get(v_zp) | (mem_get(v_zp+1) << 8);
}

/* ======================================================================= */

// Add with carry 
function i_ADC(val) {
  var toAdd = val + c;
  var signSame = !((a ^ toAdd) & 0x80);
  a = a + toAdd;

  // Carry flag processing
  c = a > 0xFF;
  a &= 0xFF;
  
  // Overflow can happen only when the signs matched originally
  v = signSame && ((a ^ toAdd) & 0x80);

  // Finish up  
  nz = a;
}

// Shift left (high bit goes into carry)
function i_ASL(p) {
  nz = mem_get(p);
  c = nz > 127;
  nz = (nz << 1) & 0xFF;
  mem_set[p](p, nz);
}

// Get bits from memory
function i_BIT(val) {
  v = val & 0x40;  // overflow bit
  nz = val & 0x80; // negative bit
  if (a & val)
    nz |= 1;       // nonzero
  else
    nz |= 0x200;   // special Z override
}

// Test and reset bits
function i_TRB(ptr) {
  nz &= 0x80; // retain old negative bit
  var val = mem_get(ptr);
  if (a & val)
    nz |= 1;       // nonzero
  else
    nz |= 0x200;   // special Z override
  mem_set[ptr](ptr, val & ~a);
}

// Test and set bits
function i_TSB(ptr) {
  nz &= 0x80; // retain old negative bit
  var val = mem_get(ptr);
  if (a & val)
    nz |= 1;       // nonzero
  else
    nz |= 0x200;   // special Z override
  mem_set[ptr](ptr, val | a);
}

// Take a branch
function i_BR() {
  var offset = mem_get(pc++);
  var newpc = pc + ((offset < 128) ? offset : offset-256);
  if ((newpc & 0xFF00) == (pc & 0xFF00))
    t += 3;
  else
    t += 4;
  pc = newpc;
  branched = true;  
}

// Break instruction
function i_BRK() {
  // Push pc+1 as per JSR
  mem_set[s+0x100](s+0x100, (pc+1) >> 8);
  s = (s-1) & 0xFF;
  mem_set[s+0x100](s+0x100, (pc+1) & 0xFF);
  s = (s-1) & 0xFF;
  // Push processor status
  i_PHP(); // 3 cycles...
  t += 4;  // ... total of 7 cycles
  // Set interrupt inhibit
  fi = true;
  // Jump to break vector
  pc = 0xFFFE;
  i_JMP();
}

// Compare register with value
function i_CMP(reg, val) {
  nz = reg + (val ^ 0xFF) + 1;
  c = nz > 0xFF;
  nz &= 0xFF;
}

// Decrement memory
function i_DEC(p) {
  nz = (mem_get(p) + 0xFF) & 0xFF;
  mem_set[p](p, nz);
}

// Jump
function i_JMP() {
  pc = mem_get(pc) | (mem_get(pc+1) << 8);
  branched = true;  
}

// Jump to subroutine
function i_JSR() {
  mem_set[s+0x100](s+0x100, (pc+1) >> 8);
  s = (s-1) & 0xFF;
  mem_set[s+0x100](s+0x100, (pc+1) & 0xFF);
  s = (s-1) & 0xFF;
  pc = mem_get(pc) | (mem_get(pc+1) << 8);
  branched = true;  
  t += 6;
}

// Increment memory
function i_INC(p) {
  nz = (mem_get(p) + 1) & 0xFF;
  mem_set[p](p, nz);
}

// Shift right (low bit goes into carry)
function i_LSR(p) {
  nz = mem_get(p);
  c = nz & 1;
  nz >>= 1;
  mem_set[p](p, nz);
}

// Translate flags to a binary p-register value
function calcPreg()
{
  var p = 0;
  if (nz & 0x80)            p |= 0x80; // N
  if (v)                    p |= 0x40; // V
  p |= 0x30;                           // unused flag, B
  if (fd)                   p |= 8;    // D
  if (fi)                   p |= 4;    // I
  if ((nz & 0x200) || !nz)  p |= 2;    // Z
  p |= c;                              // C
  return p;
}

// Push processor status bits
function i_PHP() {
  var p = calcPreg();
  mem_set[s+0x100](s+0x100, p);
  s = (s-1) & 0xFF;
  t += 3;
}

// Pop processor status bits
function i_PLP() {
  s = (s+1) & 0xFF;
  var p = mem_get(s+0x100);
  
  nz = p & 0x80;
  v  = !!(p & 0x40);
  fd = !!(p & 8);
  fi = !!(p & 4);
  if (nz && (p & 2))
    nz |= 0x200; // special Z override
  else if (!nz && !(p & 2))
    nz |= 1; // force non-zero
  c = p & 1;
  t += 4;
}

// Rotate left (high bit goes into carry)
function i_ROL(p) {
  nz = mem_get(p);
  nz = (nz << 1) | c;
  c = nz > 0xFF;
  nz &= 0xFF;
  mem_set[p](p, nz);
}

// Rotate right (low bit goes into carry)
function i_ROR(p) {
  nz = mem_get(p);
  newc = nz & 1;
  nz = ((nz>>1) | (c<<7)) & 0xFF;
  c = newc;
  mem_set[p](p, nz);
}

// Return from interrupt
function i_RTI() {
  i_PLP(); // 4 cycles...
  t += 2;  // ... total of 6 cycles
  s = (s+1) & 0xFF;
  pc = mem_get(s+0x100); // lo byte
  s = (s+1) & 0xFF;
  pc |= (mem_get(s+0x100) << 8); // hi byte
  branched = true;  
}

// Return from subroutine
function i_RTS() {
  s = (s+1) & 0xFF;
  pc = mem_get(s+0x100); // lo byte
  s = (s+1) & 0xFF;
  pc |= (mem_get(s+0x100) << 8); // hi byte
  pc++;
  branched = true;  
  t += 6;
}

// Subtract is the same as adding one's complement  
function i_SBC(val, nCyc) {
  i_ADC(val ^ 0xff, nCyc);
}

// Store val to memory
function i_ST(val, p) {
  mem_set[p](p, val);
}

/* ======================================================================= */

// Named functions for each value 6502 op; the names are useful for profiling.

function func0x69() { i_ADC(mem_get(pc++)); t += 2 }
function func0x65() { i_ADC(mem_get(mem_get(pc++))); t += 3}
function func0x75() { i_ADC(v_zp_n(x)) }
function func0x6D() { i_ADC(v_abs()) }
function func0x7D() { i_ADC(v_abs_n(x)) }
function func0x79() { i_ADC(v_abs_n(y)) }
function func0x61() { i_ADC(v_ind_x()) }
function func0x71() { i_ADC(v_ind_y()) }
function func0x29() { nz = a = (a & mem_get(pc++)); t += 2 }
function func0x25() { nz = a = (a & v_zp()) }
function func0x35() { nz = a = (a & v_zp_n(x)) }
function func0x2D() { nz = a = (a & v_abs()) }
function func0x3D() { nz = a = (a & v_abs_n(x)) }
function func0x39() { nz = a = (a & v_abs_n(y)) }
function func0x21() { nz = a = (a & v_ind_x()) }
function func0x31() { nz = a = (a & v_ind_y()) }
function func0x0A() { c = a > 127; nz = a = (a<<1)&0xff; t += 2 }
function func0x06() { i_ASL(p_zp()) }
function func0x16() { i_ASL(p_zp_n(x)) }
function func0x0E() { i_ASL(p_abs()) }
function func0x1E() { i_ASL(p_abs_n(x)) }
function func0x90() { c ? (t+=2, pc++) : i_BR() }
function func0xB0() { c ? i_BR() : (t+=2, pc++) }
function func0x30() { (nz & 0x80) ? i_BR() : (t+=2, pc++) }
function func0x10() { (nz & 0x80) ? (t+=2, pc++) : i_BR() }
function func0xD0() { ((nz & 0x200) || !nz) ? (t+=2, pc++) : i_BR() }
function func0xF0() { ((nz & 0x200) || !nz) ? i_BR() : (t+=2, pc++) }
function func0x00() { i_BRK() }
function func0x50() { v ? (t+=2, pc++) : i_BR() }
function func0x70() { v ? i_BR() : (t+=2, pc++) }
function func0x24() { i_BIT(v_zp()) }
function func0x2C() { i_BIT(v_abs()) }
function func0x18() { c = false; t += 2 }
function func0xD8() { fd = false; t += 2 }
function func0x58() { fi = false; t += 2 }
function func0xB8() { v = false; t += 2 }
function func0xC9() { i_CMP(a, mem_get(pc++)); t += 2 }
function func0xC5() { i_CMP(a, mem_get(mem_get(pc++))); t += 3 }
function func0xD5() { i_CMP(a, v_zp_n(x)) }
function func0xCD() { i_CMP(a, v_abs()) }
function func0xDD() { i_CMP(a, v_abs_n(x)) }
function func0xD9() { i_CMP(a, v_abs_n(y)) }
function func0xC1() { i_CMP(a, v_ind_x()) }
function func0xD1() { i_CMP(a, v_ind_y()) }
function func0xE0() { i_CMP(x, mem_get(pc++)); t += 2 }
function func0xE4() { i_CMP(x, mem_get(mem_get(pc++))); t += 3 }
function func0xEC() { i_CMP(x, v_abs()) }
function func0xC0() { i_CMP(y, mem_get(pc++)); t += 2 }
function func0xC4() { i_CMP(y, mem_get(mem_get(pc++))); t += 3 }
function func0xCC() { i_CMP(y, v_abs()) }
function func0xC6() { i_DEC(p_zp()) }
function func0xD6() { i_DEC(p_zp_n(x)) }
function func0xCE() { i_DEC(p_abs()) }
function func0xDE() { i_DEC(p_abs_n(x)) }
function func0xCA() { nz = x = (x-1) & 0xFF; t += 2 }
function func0x88() { nz = y = (y-1) & 0xFF; t += 2 }
function func0x49() { nz = a = (a ^ mem_get(pc++)); t += 2 }
function func0x45() { nz = a = (a ^ v_zp()) }
function func0x55() { nz = a = (a ^ v_zp_n(x)) }
function func0x4D() { nz = a = (a ^ v_abs()) }
function func0x5D() { nz = a = (a ^ v_abs_n(x)) }
function func0x59() { nz = a = (a ^ v_abs_n(y)) }
function func0x41() { nz = a = (a ^ v_ind_x()) }
function func0x51() { nz = a = (a ^ v_ind_y()) }
function func0xE6() { i_INC(p_zp()) }
function func0xF6() { i_INC(p_zp_n(x)) }
function func0xEE() { i_INC(p_abs()) }
function func0xFE() { i_INC(p_abs_n(x)) }
function func0xE8() { nz = x = (x+1) & 0xFF; t += 2 }
function func0xC8() { nz = y = (y+1) & 0xFF; t += 2 }
function func0x4C() { i_JMP(); t += 3 }
function func0x6C() { i_JMP(); i_JMP(); t += 5 }
function func0x20() { i_JSR() }
function func0xA9() { nz = a = mem_get(pc++); t += 2 }
function func0xA5() { nz = a = mem_get(mem_get(pc++)); t += 3 }
function func0xB5() { nz = a = v_zp_n(x) }
function func0xAD() { nz = a = v_abs() }
function func0xBD() { nz = a = v_abs_n(x) }
function func0xB9() { nz = a = v_abs_n(y) }
function func0xA1() { nz = a = v_ind_x() }
function func0xB1() { nz = a = v_ind_y() }
function func0xA2() { nz = x = mem_get(pc++); t += 2 }
function func0xA6() { nz = x = mem_get(mem_get(pc++)); t += 3 }
function func0xB6() { nz = x = v_zp_n(y) }
function func0xAE() { nz = x = v_abs() }
function func0xBE() { nz = x = v_abs_n(y) }
function func0xA0() { nz = y = mem_get(pc++); t += 2 }
function func0xA4() { nz = y = mem_get(mem_get(pc++)); t += 3 }
function func0xB4() { nz = y = v_zp_n(x) }
function func0xAC() { nz = y = v_abs() }
function func0xBC() { nz = y = v_abs_n(x) }
function func0x4A() { c = a & 1; nz = a = (a>>1); t += 2 }
function func0x46() { i_LSR(p_zp()) }
function func0x56() { i_LSR(p_zp_n(x)) }
function func0x4E() { i_LSR(p_abs()) }
function func0x5E() { i_LSR(p_abs_n(x)) }
function func0xEA() { t += 2 }
function func0x09() { nz = a = (a | mem_get(pc++)); t += 2 }
function func0x05() { nz = a = (a | v_zp()) }
function func0x15() { nz = a = (a | v_zp_n(x)) }
function func0x0D() { nz = a = (a | v_abs()) }
function func0x1D() { nz = a = (a | v_abs_n(x)) }
function func0x19() { nz = a = (a | v_abs_n(y)) }
function func0x01() { nz = a = (a | v_ind_x()) }
function func0x11() { nz = a = (a | v_ind_y()) }
function func0x48() { mem_set[s+0x100](s+0x100, a); s = (s-1) & 0xFF; t += 3 }
function func0x08() { i_PHP() }
function func0x68() { s = (s+1) & 0xFF; nz = a = mem_get(s+0x100); t += 4 }
function func0x28() { i_PLP() }
function func0x2A() { a = ((a<<1) + c); c = a > 0xFF; nz = a = (a & 0xFF); t += 2 }
function func0x26() { i_ROL(p_zp()) }
function func0x36() { i_ROL(p_zp_n(x)) }
function func0x2E() { i_ROL(p_abs()) }
function func0x3E() { i_ROL(p_abs_n(x)) }
function func0x6A() { var newc = a & 1; nz = a = (a>>1) + (c<<7); c = newc; t += 2 }
function func0x66() { i_ROR(p_zp()) }
function func0x76() { i_ROR(p_zp_n(x)) }
function func0x6E() { i_ROR(p_abs()) }
function func0x7E() { i_ROR(p_abs_n(x)) }
function func0x40() { i_RTI() }
function func0x60() { i_RTS() }
function func0xE9() { i_SBC(mem_get(pc++)); t += 2 }
function func0xE5() { i_SBC(mem_get(mem_get(pc++))); t += 3}
function func0xF5() { i_SBC(v_zp_n(x)) }
function func0xED() { i_SBC(v_abs()) }
function func0xFD() { i_SBC(v_abs_n(x)) }
function func0xF9() { i_SBC(v_abs_n(y)) }
function func0xE1() { i_SBC(v_ind_x()) }
function func0xF1() { i_SBC(v_ind_y()) }
function func0x38() { c = true; t += 2 }
function func0xF8() { fd = true; t += 2 }
function func0x78() { fi = true; t += 2 }
function func0x85() { i_ST(a, s_zp()) }
function func0x95() { i_ST(a, s_zp_n(x)) }
function func0x8D() { i_ST(a, s_abs()) }
function func0x9D() { i_ST(a, s_abs_n(x)) }
function func0x99() { i_ST(a, s_abs_n(y)) }
function func0x81() { i_ST(a, s_ind_x()) }
function func0x91() { i_ST(a, s_ind_y()) }
function func0x86() { i_ST(x, s_zp()) }
function func0x96() { i_ST(x, s_zp_n(y)) }
function func0x8E() { i_ST(x, s_abs()) }
function func0x84() { i_ST(y, s_zp()) }
function func0x94() { i_ST(y, s_zp_n(x)) }
function func0x8C() { i_ST(y, s_abs()) }
function func0xAA() { nz = x = a; t += 2 }
function func0xA8() { nz = y = a; t += 2 }
function func0xBA() { nz = x = s; t += 2 }
function func0x8A() { nz = a = x; t += 2 }
function func0x9A() { nz = s = x; t += 2 }
function func0x98() { nz = a = y; t += 2 }

//////////////////////////////////////////////////////////////////////////////////////
// Additions for 65c02.
// 1. (zp) addressing mode
function func0x72() { i_ADC(v_ind()) }                                         // ADC
function func0x32() { nz = a = (a & v_ind()) }                                 // AND
function func0xD2() { i_CMP(v_ind()) }                                         // CMP
function func0x52() { nz = a = (a ^ v_ind()) }                                 // EOR
function func0xB2() { nz = a = v_ind() }                                       // LDA
function func0x12() { nz = a = (a | v_ind()) }                                 // ORA
function func0xF2() { i_SBC(v_ind()) }                                         // SBC
function func0x92() { i_ST(a, s_ind()) }                                       // STA
// 2. More addressing modes for BIT
function func0x89() { i_BIT(mem_get(pc++)); t += 2 }                           // BIT
function func0x34() { i_BIT(v_zp_n(x)) }                                       // "
function func0x3C() { i_BIT(v_abs_n(x)) }                                      // "
// 3. DEC/INC accumulator
function func0x3A() { nz = a = (a-1) & 0xFF; t += 2 }                          // DEC
function func0x1A() { nz = a = (a+1) & 0xFF; t += 2 }                          // INC
// 4. JMP (abs,x) mode
function func0x7C() { i_JMP(); pc += x; i_JMP(); t += 6 }
// 5. Additional instructions
function func0x80() { i_BR() }                                                 // BRA
function func0xDA() { mem_set[s+0x100](s+0x100, x); s = (s-1) & 0xFF; t += 3 } // PHX
function func0x5A() { mem_set[s+0x100](s+0x100, y); s = (s-1) & 0xFF; t += 3 } // PHY
function func0xFA() { s = (s+1) & 0xFF; nz = x = mem_get(s+0x100); t += 4 }    // PLX
function func0x7A() { s = (s+1) & 0xFF; nz = y = mem_get(s+0x100); t += 4 }    // PLY
function func0x64() { i_ST(0, s_zp()) }                                        // STZ
function func0x74() { i_ST(0, s_zp_n(x)) }                                     // "
function func0x9C() { i_ST(0, s_abs()) }                                       // "
function func0x9E() { i_ST(0, s_abs_n(x)) }                                    // "
function func0x14() { i_TRB(p_zp()) }                                          // TRB
function func0x1C() { i_TRB(p_abs()) }                                         // "
function func0x04() { i_TSB(p_abs()) }                                         // TSB
function func0x0C() { i_TSB(p_abs()) }                                         // "
// 6. Unused opcodes - explicit size and timing
function func0x02() { ++pc; t += 2 }
function func0x03() { t += 1 }
function func0x07() { t += 1 }
function func0x0B() { t += 1 }
function func0x0F() { t += 1 }
function func0x13() { t += 1 }
function func0x17() { t += 1 }
function func0x1B() { t += 1 }
function func0x1F() { t += 1 }
function func0x22() { ++pc; t += 2 }
function func0x23() { t += 1 }
function func0x27() { t += 1 }
function func0x2B() { t += 1 }
function func0x2F() { t += 1 }
function func0x33() { t += 1 }
function func0x37() { t += 1 }
function func0x3B() { t += 1 }
function func0x3F() { t += 1 }
function func0x42() { ++pc; t += 2 }
function func0x43() { t += 1 }
function func0x44() { ++pc; t += 3 }
function func0x47() { t += 1 }
function func0x4B() { t += 1 }
function func0x4F() { t += 1 }
function func0x53() { t += 1 }
function func0x54() { ++pc; t += 4 }
function func0x57() { t += 1 }
function func0x5B() { t += 1 }
function func0x5C() { pc += 2; t += 8 }
function func0x5F() { t += 1 }
function func0x62() { ++pc; t += 2 }
function func0x63() { t += 1 }
function func0x67() { t += 1 }
function func0x6B() { t += 1 }
function func0x6F() { t += 1 }
function func0x73() { t += 1 }
function func0x77() { t += 1 }
function func0x7B() { t += 1 }
function func0x7F() { t += 1 }
function func0x82() { ++pc; t += 2 }
function func0x83() { t += 1 }
function func0x87() { t += 1 }
function func0x8B() { t += 1 }
function func0x8F() { t += 1 }
function func0x93() { t += 1 }
function func0x97() { t += 1 }
function func0x9B() { t += 1 }
function func0x9F() { t += 1 }
function func0xA3() { t += 1 }
function func0xA7() { t += 1 }
function func0xAB() { t += 1 }
function func0xAF() { t += 1 }
function func0xB3() { t += 1 }
function func0xB7() { t += 1 }
function func0xBB() { t += 1 }
function func0xBF() { t += 1 }
function func0xC2() { ++pc; t += 2 }
function func0xC3() { t += 1 }
function func0xC7() { t += 1 }
function func0xCB() { t += 1 }
function func0xCF() { t += 1 }
function func0xD3() { t += 1 }
function func0xD4() { ++pc; t += 4 }
function func0xD7() { t += 1 }
function func0xDB() { t += 1 }
function func0xDC() { pc += 2; t += 4 }
function func0xDF() { t += 1 }
function func0xE2() { ++pc; t += 2 }
function func0xE3() { t += 1 }
function func0xE7() { t += 1 }
function func0xEB() { t += 1 }
function func0xEF() { t += 1 }
function func0xF3() { t += 1 }
function func0xF4() { ++pc; t += 4 }
function func0xF7() { t += 1 }
function func0xFB() { t += 1 }
function func0xFC() { pc += 2; t += 4 }
function func0xFF() { t += 1 }

// Quick-access table of opcode to function
var opTbl = {
  0x69: func0x69,
  0x65: func0x65,
  0x75: func0x75,
  0x6D: func0x6D,
  0x7D: func0x7D,
  0x79: func0x79,
  0x61: func0x61,
  0x71: func0x71,
  0x29: func0x29,
  0x25: func0x25,
  0x35: func0x35,
  0x2D: func0x2D,
  0x3D: func0x3D,
  0x39: func0x39,
  0x21: func0x21,
  0x31: func0x31,
  0x0A: func0x0A,
  0x06: func0x06,
  0x16: func0x16,
  0x0E: func0x0E,
  0x1E: func0x1E,
  0x90: func0x90,
  0xB0: func0xB0,
  0x30: func0x30,
  0x10: func0x10,
  0xD0: func0xD0,
  0xF0: func0xF0,
  0x00: func0x00,
  0x50: func0x50,
  0x70: func0x70,
  0x24: func0x24,
  0x2C: func0x2C,
  0x18: func0x18,
  0xD8: func0xD8,
  0x58: func0x58,
  0xB8: func0xB8,
  0xC9: func0xC9,
  0xC5: func0xC5,
  0xD5: func0xD5,
  0xCD: func0xCD,
  0xDD: func0xDD,
  0xD9: func0xD9,
  0xC1: func0xC1,
  0xD1: func0xD1,
  0xE0: func0xE0,
  0xE4: func0xE4,
  0xEC: func0xEC,
  0xC0: func0xC0,
  0xC4: func0xC4,
  0xCC: func0xCC,
  0xC6: func0xC6,
  0xD6: func0xD6,
  0xCE: func0xCE,
  0xDE: func0xDE,
  0xCA: func0xCA,
  0x88: func0x88,
  0x49: func0x49,
  0x45: func0x45,
  0x55: func0x55,
  0x4D: func0x4D,
  0x5D: func0x5D,
  0x59: func0x59,
  0x41: func0x41,
  0x51: func0x51,
  0xE6: func0xE6,
  0xF6: func0xF6,
  0xEE: func0xEE,
  0xFE: func0xFE,
  0xE8: func0xE8,
  0xC8: func0xC8,
  0x4C: func0x4C,
  0x6C: func0x6C,
  0x20: func0x20,
  0xA9: func0xA9,
  0xA5: func0xA5,
  0xB5: func0xB5,
  0xAD: func0xAD,
  0xBD: func0xBD,
  0xB9: func0xB9,
  0xA1: func0xA1,
  0xB1: func0xB1,
  0xA2: func0xA2,
  0xA6: func0xA6,
  0xB6: func0xB6,
  0xAE: func0xAE,
  0xBE: func0xBE,
  0xA0: func0xA0,
  0xA4: func0xA4,
  0xB4: func0xB4,
  0xAC: func0xAC,
  0xBC: func0xBC,
  0x4A: func0x4A,
  0x46: func0x46,
  0x56: func0x56,
  0x4E: func0x4E,
  0x5E: func0x5E,
  0xEA: func0xEA,
  0x09: func0x09,
  0x05: func0x05,
  0x15: func0x15,
  0x0D: func0x0D,
  0x1D: func0x1D,
  0x19: func0x19,
  0x01: func0x01,
  0x11: func0x11,
  0x48: func0x48,
  0x08: func0x08,
  0x68: func0x68,
  0x28: func0x28,
  0x2A: func0x2A,
  0x26: func0x26,
  0x36: func0x36,
  0x2E: func0x2E,
  0x3E: func0x3E,
  0x6A: func0x6A,
  0x66: func0x66,
  0x76: func0x76,
  0x6E: func0x6E,
  0x7E: func0x7E,
  0x40: func0x40,
  0x60: func0x60,
  0xE9: func0xE9,
  0xE5: func0xE5,
  0xF5: func0xF5,
  0xED: func0xED,
  0xFD: func0xFD,
  0xF9: func0xF9,
  0xE1: func0xE1,
  0xF1: func0xF1,
  0x38: func0x38,
  0xF8: func0xF8,
  0x78: func0x78,
  0x85: func0x85,
  0x95: func0x95,
  0x8D: func0x8D,
  0x9D: func0x9D,
  0x99: func0x99,
  0x81: func0x81,
  0x91: func0x91,
  0x86: func0x86,
  0x96: func0x96,
  0x8E: func0x8E,
  0x84: func0x84,
  0x94: func0x94,
  0x8C: func0x8C,
  0xAA: func0xAA,
  0xA8: func0xA8,
  0xBA: func0xBA,
  0x8A: func0x8A,
  0x9A: func0x9A,
  0x98: func0x98,
  // 65c02 additions
  0x72: func0x72,
  0x32: func0x32,
  0xD2: func0xD2,
  0x52: func0x52,
  0xB2: func0xB2,
  0x12: func0x12,
  0xF2: func0xF2,
  0x92: func0x92,
  0x89: func0x89,
  0x34: func0x34,
  0x3C: func0x3C,
  0x3A: func0x3A,
  0x1A: func0x1A,
  0x7C: func0x7C,
  0x80: func0x80,
  0xDA: func0xDA,
  0x5A: func0x5A,
  0xFA: func0xFA,
  0x7A: func0x7A,
  0x64: func0x64,
  0x74: func0x74,
  0x9C: func0x9C,
  0x9E: func0x9E,
  0x14: func0x14,
  0x1C: func0x1C,
  0x04: func0x04,
  0x0C: func0x0C,
  0x02: func0x02,
  0x03: func0x03,
  0x07: func0x07,
  0x0B: func0x0B,
  0x0F: func0x0F,
  0x13: func0x13,
  0x17: func0x17,
  0x1B: func0x1B,
  0x1F: func0x1F,
  0x22: func0x22,
  0x23: func0x23,
  0x27: func0x27,
  0x2B: func0x2B,
  0x2F: func0x2F,
  0x33: func0x33,
  0x37: func0x37,
  0x3B: func0x3B,
  0x3F: func0x3F,
  0x42: func0x42,
  0x43: func0x43,
  0x44: func0x44,
  0x47: func0x47,
  0x4B: func0x4B,
  0x4F: func0x4F,
  0x53: func0x53,
  0x54: func0x54,
  0x57: func0x57,
  0x5B: func0x5B,
  0x5C: func0x5C,
  0x5F: func0x5F,
  0x62: func0x62,
  0x63: func0x63,
  0x67: func0x67,
  0x6B: func0x6B,
  0x6F: func0x6F,
  0x73: func0x73,
  0x77: func0x77,
  0x7B: func0x7B,
  0x7F: func0x7F,
  0x82: func0x82,
  0x83: func0x83,
  0x87: func0x87,
  0x8B: func0x8B,
  0x8F: func0x8F,
  0x93: func0x93,
  0x97: func0x97,
  0x9B: func0x9B,
  0x9F: func0x9F,
  0xA3: func0xA3,
  0xA7: func0xA7,
  0xAB: func0xAB,
  0xAF: func0xAF,
  0xB3: func0xB3,
  0xB7: func0xB7,
  0xBB: func0xBB,
  0xBF: func0xBF,
  0xC2: func0xC2,
  0xC3: func0xC3,
  0xC7: func0xC7,
  0xCB: func0xCB,
  0xCF: func0xCF,
  0xD3: func0xD3,
  0xD4: func0xD4,
  0xD7: func0xD7,
  0xDB: func0xDB,
  0xDC: func0xDC,
  0xDF: func0xDF,
  0xE2: func0xE2,
  0xE3: func0xE3,
  0xE7: func0xE7,
  0xEB: func0xEB,
  0xEF: func0xEF,
  0xF3: func0xF3,
  0xF4: func0xF4,
  0xF7: func0xF7,
  0xFB: func0xFB,
  0xFC: func0xFC,
  0xFF: func0xFF,
};

// Op strings for debugging
var opStrings = {
  0x69: "ADC #imm",
  0x65: "ADC zp",
  0x75: "ADC zp,x",
  0x6D: "ADC abs",
  0x7D: "ADC abs,x",
  0x79: "ADC abs,y",
  0x61: "ADC (ind,x)",
  0x71: "ADC (ind),y",
  0x29: "AND #imm",
  0x25: "AND zp",
  0x35: "AND zp,x",
  0x2D: "AND abs",
  0x3D: "AND abs,x",
  0x39: "AND abs,y",
  0x21: "AND (ind,x)",
  0x31: "AND (ind),y",
  0x0A: "ASL A",
  0x06: "ASL zp",
  0x16: "ASL zp,x",
  0x0E: "ASL abs",
  0x1E: "ASL abs,x",
  0x90: "BCC",
  0xB0: "BCS",
  0x30: "BMI",
  0x10: "BPL",
  0xD0: "BNE",
  0xF0: "BEQ",
  0x00: "BRK",
  0x50: "BVC",
  0x70: "BVS",
  0x24: "BIT zp",
  0x2C: "BIT abs",
  0x18: "CLC",
  0xD8: "CLD",
  0x58: "CLI",
  0xB8: "CLV",
  0xC9: "CMP #imm",
  0xC5: "CMP zp",
  0xD5: "CMP zp,x",
  0xCD: "CMP abs",
  0xDD: "CMP abs,x",
  0xD9: "CMP abs,y",
  0xC1: "CMP (ind,x)",
  0xD1: "CMP (ind),y",
  0xE0: "CPX #imm",
  0xE4: "CPX zp",
  0xEC: "CPX abs",
  0xC0: "CPY #imm",
  0xC4: "CPY zp",
  0xCC: "CPY abs",
  0xC6: "DEC zp",
  0xD6: "DEC zp,x",
  0xCE: "DEC abs",
  0xDE: "DEC abs,x",
  0xCA: "DEX",
  0x88: "DEY",
  0x49: "EOR #imm",
  0x45: "EOR zp",
  0x55: "EOR zp,x",
  0x4D: "EOR abs",
  0x5D: "EOR abs,x",
  0x59: "EOR abs,y",
  0x41: "EOR (ind,x)",
  0x51: "EOR (ind),y",
  0xE6: "INC zp",
  0xF6: "INC zp,x",
  0xEE: "INC abs",
  0xFE: "INC abs,x",
  0xE8: "INX",
  0xC8: "INY",
  0x4C: "JMP abs",
  0x6C: "JMP (ind)",
  0x20: "JSR abs",
  0xA9: "LDA #imm",
  0xA5: "LDA zp",
  0xB5: "LDA zp,x",
  0xAD: "LDA abs",
  0xBD: "LDA abs,x",
  0xB9: "LDA abs,y",
  0xA1: "LDA (ind,x)",
  0xB1: "LDA (ind),y",
  0xA2: "LDX #imm",
  0xA6: "LDX zp",
  0xB6: "LDX zp,y",
  0xAE: "LDX abs",
  0xBE: "LDX abs,y",
  0xA0: "LDY #imm",
  0xA4: "LDY zp",
  0xB4: "LDY zp,x",
  0xAC: "LDY abs",
  0xBC: "LDY abs,x",
  0x4A: "LSR A",
  0x46: "LSR zp",
  0x56: "LSR zp,x",
  0x4E: "LSR abs",
  0x5E: "LSR abs,x",
  0xEA: "NOP",
  0x09: "ORA #imm",
  0x05: "ORA zp",
  0x15: "ORA zp,x",
  0x0D: "ORA abs",
  0x1D: "ORA abs,x",
  0x19: "ORA abs,y",
  0x01: "ORA (ind,x)",
  0x11: "ORA (ind),y",
  0x48: "PHA",
  0x08: "PHP",
  0x68: "PLA",
  0x28: "PLP",
  0x2A: "ROL",
  0x26: "ROL zp",
  0x36: "ROL zp,x",
  0x2E: "ROL abs",
  0x3E: "ROL abs,x",
  0x6A: "ROR",
  0x66: "ROR zp",
  0x76: "ROR zp,x",
  0x6E: "ROR abs",
  0x7E: "ROR abs,x",
  0x40: "RTI",
  0x60: "RTS",
  0xE9: "SBC #imm",
  0xE5: "SBC zp",
  0xF5: "SBC zp,x",
  0xED: "SBC abs",
  0xFD: "SBC abs,x",
  0xF9: "SBC abs,y",
  0xE1: "SBC (ind,x)",
  0xF1: "SBC (ind),y",
  0x38: "SEC",
  0xF8: "SED",
  0x78: "SEI",
  0x85: "STA zp",
  0x95: "STA zp,x",
  0x8D: "STA abs",
  0x9D: "STA abs,x",
  0x99: "STA abs,y",
  0x81: "STA (ind,x)",
  0x91: "STA (ind),y",
  0x86: "STX zp",
  0x96: "STX zp,y",
  0x8E: "STX abs",
  0x84: "STY zp",
  0x94: "STY zp,x",
  0x8C: "STY abs",
  0xAA: "TAX",
  0xA8: "TAY",
  0xBA: "TSX",
  0x8A: "TXA",
  0x9A: "TXS",
  0x98: "TYA",
  // Extra 65c02 instructions
  0x72: "ADC (ind)",
  0x32: "AND (ind)",
  0xD2: "CMP (ind)",
  0x52: "EOR (ind)",
  0xB2: "LDA (ind)",
  0x12: "ORA (ind)",
  0xF2: "SBC (ind)",
  0x92: "STA (ind)",
  0x89: "BIT imm",
  0x34: "BIT zp,x",
  0x3C: "BIT abs,x",
  0x3A: "DEC a",
  0x1A: "INC a",
  0x7C: "JMP (ind)",
  0x80: "BRA",
  0xDA: "PHX",
  0x5A: "PHY",
  0xFA: "PLX",
  0xFA: "PLY",
  0x64: "STZ zp",
  0x74: "STZ zp,x",
  0x9C: "STZ abs",
  0x9E: "STZ abs,x",
  0x14: "TRB zp",
  0x1C: "TRB abs",
  0x04: "TSB zp",
  0x0C: "TSB abs",
  // 6. Unused opcodes - explicit size and timing
  0x02: "NOP_2_2",
  0x03: "NOP_1_1",
  0x07: "NOP_1_1",
  0x0B: "NOP_1_1",
  0x0F: "NOP_1_1",
  0x13: "NOP_1_1",
  0x17: "NOP_1_1",
  0x1B: "NOP_1_1",
  0x1F: "NOP_1_1",
  0x22: "NOP_2_2",
  0x23: "NOP_1_1",
  0x27: "NOP_1_1",
  0x2B: "NOP_1_1",
  0x2F: "NOP_1_1",
  0x33: "NOP_1_1",
  0x37: "NOP_1_1",
  0x3B: "NOP_1_1",
  0x3F: "NOP_1_1",
  0x42: "NOP_2_2",
  0x43: "NOP_1_1",
  0x44: "NOP_2_3",
  0x47: "NOP_1_1",
  0x4B: "NOP_1_1",
  0x4F: "NOP_1_1",
  0x53: "NOP_1_1",
  0x54: "NOP_2_4",
  0x57: "NOP_1_1",
  0x5B: "NOP_1_1",
  0x5C: "NOP_3_8",
  0x5F: "NOP_1_1",
  0x62: "NOP_2_2",
  0x63: "NOP_1_1",
  0x67: "NOP_1_1",
  0x6B: "NOP_1_1",
  0x6F: "NOP_1_1",
  0x73: "NOP_1_1",
  0x77: "NOP_1_1",
  0x7B: "NOP_1_1",
  0x7F: "NOP_1_1",
  0x82: "NOP_2_2",
  0x83: "NOP_1_1",
  0x87: "NOP_1_1",
  0x8B: "NOP_1_1",
  0x8F: "NOP_1_1",
  0x93: "NOP_1_1",
  0x97: "NOP_1_1",
  0x9B: "NOP_1_1",
  0x9F: "NOP_1_1",
  0xA3: "NOP_1_1",
  0xA7: "NOP_1_1",
  0xAB: "NOP_1_1",
  0xAF: "NOP_1_1",
  0xB3: "NOP_1_1",
  0xB7: "NOP_1_1",
  0xBB: "NOP_1_1",
  0xBF: "NOP_1_1",
  0xC2: "NOP_2_2",
  0xC3: "NOP_1_1",
  0xC7: "NOP_1_1",
  0xCB: "NOP_1_1",
  0xCF: "NOP_1_1",
  0xD3: "NOP_1_1",
  0xD4: "NOP_2_4",
  0xD7: "NOP_1_1",
  0xDB: "NOP_1_1",
  0xDC: "NOP_3_4",
  0xDF: "NOP_1_1",
  0xE2: "NOP_2_2",
  0xE3: "NOP_1_1",
  0xE7: "NOP_1_1",
  0xEB: "NOP_1_1",
  0xEF: "NOP_1_1",
  0xF3: "NOP_1_1",
  0xF4: "NOP_2_4",
  0xF7: "NOP_1_1",
  0xFB: "NOP_1_1",
  0xFC: "NOP_3_4",
  0xFF: "NOP_1_1",
};
  

